{ This unit defines the structure of the model. There are four functions. The
  first function, called counts, defines the number, names, and units of the
  model, the state variables, the process variables, the driver variables and
  the parameters. The second function, called processes, is the actual equations
  which make up the model. The third function, derivs, calculates the
  derivatives of state variables. And the fourth function, parcount, is used to
  automatically number the parameters consecutively. 
    The state variables, driver variables, process variables and parameters are
  all stored in global arrays, called stat, drive, proc, and par, respectively.
  The function counts accesses the global arrays directly but the other functions
  operate on copies of the global arrays. }
unit equations;

interface

uses  stypes, math;

PROCEDURE counts;
PROCEDURE processes(time:double; dtime:double; var tdrive:drivearray;
                       var tpar:paramarray; var tstat:statearray;
                       var tproc:processarray; CalculateDiscrete:Boolean);
PROCEDURE derivs(t, drt:double; var tdrive:drivearray; var tpar:paramarray;
             var statevalue:yValueArray; VAR dydt:yValueArray);
function ParCount(processnum:integer) : integer;

var
  tproc:processarray;
  tstat:statearray;
  sensflag:boolean;
  newyear:Boolean = false;

implementation

uses frontend, calculate, options;

           { Do not make modifcations above this line. }
{*****************************************************************************}

{ This procedure defines the model. The number of parameters, state, driver and
  process variables are all set in this procedure. The model name, version
  number and time unit are also set here. This procedure accesses the global
  arrays containing the the parameters, state, driver and process variables and
  the global structure ModelDef directly, to save memory space. }
PROCEDURE counts;
var
 i,npar,CurrentProc:integer;
begin
{ Set the modelname, version and time unit. }
ModelDef.modelname := 'MEL 1997';
ModelDef.versionnumber := '1.0.0';
ModelDef.timeunit := 'yrs';
ModelDef.contactperson := 'your name here';
ModelDef.contactaddress1 := 'address';
ModelDef.contactaddress2 := 'address';
ModelDef.contactaddress3 := 'address';

{ Set the number of state variables in the model. The maximum number of state
  variables is maxstate, in unit stypes. }
ModelDef.numstate := 7;

{ Enter the name, units and symbol for each state variable. The maximum length
  of the state variable name is 17 characters and the maximum length for units
  and symbol is stringlength (specified in unit stypes) characters. }
 
 
with stat[1] do
 begin
    name:='C: Biomass';  units:='g C m-2';  symbol:='Bc';
 end;
 
with stat[2] do
 begin
    name:='N: Biomass';  units:='g N m-2';  symbol:='Bn';
 end;
 
with stat[3] do
 begin
    name:='C: Effort';  units:='unitless';  symbol:='Vc';
 end;
 
with stat[4] do
 begin
    name:='N: Effort';  units:='unitless';  symbol:='Vn';
 end;
 
with stat[5] do
 begin
    name:='C: Detrital';  units:='g C m-2';  symbol:='Dc';
 end;
 
with stat[6] do
 begin
    name:='N: Detrital';  units:='g N m-2';  symbol:='Dn';
 end;
 
with stat[7] do
 begin
    name:='N: Inorganic';  units:='g N m-2';  symbol:='N';
 end;

{ Set the total number of processes in the model. The first numstate processes
  are the derivatives of the state variables. The maximum number of processes is
  maxparam, in unit stypes. }
ModelDef.numprocess := ModelDef.numstate + 24;

{ For each process, set proc[i].parameters equal to the number of parameters
  associated with that process, and set IsDiscrete to true or false. After each
  process, set the name, units, and symbol for all parameters associated with
  that process. Note that Parcount returns the total number of parameters in
  all previous processes. }
 
CurrentProc := ModelDef.numstate + 1;
With proc[CurrentProc] do
   begin
      name       := 'Optimum Veg C:N';
      units       := 'g C g N';
      symbol       := 'q';
      parameters       := 2;
      ptype       := ptGroup3;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Wood C:N';  units:='g C g-1 N';  symbol:='qWood';
 end;
with par[npar + 2] do
 begin
    name:='C:N half saturation';  units:='g C m-2';  symbol:='kq';
 end;
 
CurrentProc := ModelDef.numstate + 2;
With proc[CurrentProc] do
   begin
      name       := 'C relative growth rate';
      units       := 'yr-1';
      symbol       := 'GrC';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 3;
With proc[CurrentProc] do
   begin
      name       := 'N relative growth rate';
      units       := 'yr-1';
      symbol       := 'GrN';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 4;
With proc[CurrentProc] do
   begin
      name       := 'Acclimation potential';
      units       := 'unitless';
      symbol       := 'aa';
      parameters       := 1;
      ptype       := ptGroup3;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Acclim damping coefficient';  units:='yr';  symbol:='tau';
 end;
 
CurrentProc := ModelDef.numstate + 5;
With proc[CurrentProc] do
   begin
      name       := 'C org leaching loss';
      units       := 'g C m-2 yr-1';
      symbol       := 'LCd';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 6;
With proc[CurrentProc] do
   begin
      name       := 'N org leaching loss';
      units       := 'g N m-2 yr-1';
      symbol       := 'LNd';
      parameters       := 1;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Org N loss rate';  units:='day-1';  symbol:='betaD';
 end;
 
CurrentProc := ModelDef.numstate + 7;
With proc[CurrentProc] do
   begin
      name       := 'N inorg leaching loss';
      units       := 'g N m-2 day-1';
      symbol       := 'LNe';
      parameters       := 1;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Inorg N loss rate';  units:='day-1';  symbol:='betaNE';
 end;
 
CurrentProc := ModelDef.numstate + 8;
With proc[CurrentProc] do
   begin
      name       := 'C litter';
      units       := 'g C m-2 yr-1';
      symbol       := 'LCv';
      parameters       := 1;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Litterfall rate';  units:='day-1';  symbol:='mC';
 end;
 
CurrentProc := ModelDef.numstate + 9;
With proc[CurrentProc] do
   begin
      name       := 'N litter';
      units       := 'g N m-2 yr-1';
      symbol       := 'LNv';
      parameters       := 1;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Litterfall rate';  units:='day-1';  symbol:='mN';
 end;
 
CurrentProc := ModelDef.numstate + 10;
With proc[CurrentProc] do
   begin
      name       := 'Surface Area';
      units       := 'm2 m-2';
      symbol       := 'Sc';
      parameters       := 2;
      ptype       := ptGroup3;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='C surface area exp';  units:='unitless';  symbol:='zc';
 end;
with par[npar + 2] do
 begin
    name:='C surface area exp';  units:='unitless';  symbol:='zn';
 end;
 
CurrentProc := ModelDef.numstate + 11;
With proc[CurrentProc] do
   begin
      name       := 'Surface Area';
      units       := 'm2 m-2';
      symbol       := 'Sn';
      parameters       := 0;
      ptype       := ptGroup3;
   end;
 
CurrentProc := ModelDef.numstate + 12;
With proc[CurrentProc] do
   begin
      name       := 'Inorg C Uptake';
      units       := 'g C m-2 yr-1';
      symbol       := 'UCv';
      parameters       := 2;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='InorgC uptake rate';  units:='yr-1';  symbol:='gC';
 end;
with par[npar + 2] do
 begin
    name:='InorgC half saturation';  units:='g C m-2';  symbol:='kC';
 end;
 
CurrentProc := ModelDef.numstate + 13;
With proc[CurrentProc] do
   begin
      name       := 'Inorg N Uptake';
      units       := 'g N m-2 yr-1';
      symbol       := 'UNv';
      parameters       := 2;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='InorgN uptake rate';  units:='yr-1';  symbol:='gN';
 end;
with par[npar + 2] do
 begin
    name:='InorgN half saturation';  units:='g N m-2';  symbol:='kN';
 end;
 
CurrentProc := ModelDef.numstate + 14;
With proc[CurrentProc] do
   begin
      name       := 'Veg Inorg C release';
      units       := 'g C m-2 yr-1';
      symbol       := 'RCv';
      parameters       := 1;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Veg InorC release rate';  units:='yr-1';  symbol:='rC';
 end;
 
CurrentProc := ModelDef.numstate + 15;
With proc[CurrentProc] do
   begin
      name       := 'Veg Inorg N release';
      units       := 'g N m-2 yr-1';
      symbol       := 'RNv';
      parameters       := 1;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Veg InorN release rate';  units:='day-1';  symbol:='rN';
 end;
 
CurrentProc := ModelDef.numstate + 16;
With proc[CurrentProc] do
   begin
      name       := 'N Immobilization';
      units       := 'g N m-2 yr-1';
      symbol       := 'UNm';
      parameters       := 4;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Decomp rate';  units:='yr-1';  symbol:='psi';
 end;
with par[npar + 2] do
 begin
    name:='Microbe N up rate';  units:='g g-1';  symbol:='alphaN';
 end;
with par[npar + 3] do
 begin
    name:='Microbe consum ratio';  units:='g g-1';  symbol:='theta';
 end;
with par[npar + 4] do
 begin
    name:='N immob half sat';  units:='g N m-2';  symbol:='kNm';
 end;
 
CurrentProc := ModelDef.numstate + 17;
With proc[CurrentProc] do
   begin
      name       := 'Micro C efficiency';
      units       := 'unitless';
      symbol       := 'lambdaC';
      parameters       := 1;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Max Micro C effic';  units:='unitless';  symbol:='epsilonC';
 end;
 
CurrentProc := ModelDef.numstate + 18;
With proc[CurrentProc] do
   begin
      name       := 'Micro N efficiency';
      units       := 'unitless';
      symbol       := 'lambdaN';
      parameters       := 1;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Max Micro N effic';  units:='unitless';  symbol:='epsilonN';
 end;
 
CurrentProc := ModelDef.numstate + 19;
With proc[CurrentProc] do
   begin
      name       := 'Micro C consumption';
      units       := 'g C m-2 yr-1';
      symbol       := 'mmC';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 20;
With proc[CurrentProc] do
   begin
      name       := 'Micro N consumption';
      units       := 'g N m-2 yr-1';
      symbol       := 'mmN';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 21;
With proc[CurrentProc] do
   begin
      name       := 'Microbial Resp';
      units       := 'g C m-2 yr-1';
      symbol       := 'RCm';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 22;
With proc[CurrentProc] do
   begin
      name       := 'N Mineralization';
      units       := 'g N m-2 yr-1';
      symbol       := 'RNm';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 23;
With proc[CurrentProc] do
   begin
      name       := '*Effort function1';
      units       := 'unitless';
      symbol       := 'Vstar[1]';
      parameters       := 1;
      ptype       := ptGroup3;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Acclimation rate1';  units:='yr-1';  symbol:='a[1]';
 end;
 
CurrentProc := ModelDef.numstate + 24;
With proc[CurrentProc] do
   begin
      name       := '*Effort function2';
      units       := 'unitless';
      symbol       := 'Vstar[2]';
      parameters       := 1;
      ptype       := ptGroup3;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Acclimation rate2';  units:='yr-1';  symbol:='a[2]';
 end;

{ Set the total number of drivers in the model. The maximum number of drivers is
  maxdrive, in unit stypes. }
ModelDef.numdrive := 0;

{ Set the names, units, and symbols of the drivers. The maximum length for the
  name, units and symbol is 20 characters. }

{ The first numstate processes are the derivatives of the state variables. The
  code sets the names, units and symbols accordingly.}
for i:= 1 to ModelDef.numstate do proc[i].name:='d'+stat[i].name+'dt';
for i:= 1 to ModelDef.numstate do proc[i].units := stat[i].units + 't-1';
for i:= 1 to ModelDef.numstate do proc[i].symbol := 'd' + stat[i].symbol + 'dt';

{ Code to sum up the total number of parameters in the model. Do not change the
  next few lines. }
ModelDef.numparam := 0;
for i := 1 to ModelDef.NumProcess do
  ModelDef.numparam := ModelDef.numparam + proc[i].parameters;

end; // counts procedure


{ A procedure to calculate the value of all states and processes at the current
  time. This function accesses time, state variables and process variables by
  reference, ie it uses the same array as the calling routine. It does not use
  the global variables time, stat and proc because values calculated during
  integration might later be discarded. It does access the global variables par,
  drive and ModelDef directly because those values are not modified.

  The model equations are written using variable names which correspond to the
  actual name instead of using the global arrays (i.e. SoilWater instead of
  stat[7].value). This makes it necessary to switch all values into local
  variables, do all the calculations and then put everything back into the
  global variables. Lengthy but worth it in terms of readability of the code. }

// Choose either GlobalPs, ArcticPs, or none here so the appropriate Ps model is compiled below.
{$DEFINE none}

PROCEDURE processes(time:double; dtime:double; var tdrive:drivearray;
                       var tpar:paramarray; var tstat:statearray;
                       var tproc:processarray; CalculateDiscrete:Boolean);
{$IFDEF GlobalPs}
const
// Global Ps parameters
 x1 = 11.04;             x2 = 0.03;
 x5 = 0.216;             x6 = 0.6;
 x7 = 3.332;             x8 = 0.004;
 x9 = 1.549;             x10 = 1.156;
 gammastar = 0;          kCO2 = 995.4;  }
{$ENDIF}

// Modify constant above (line above "procedure processes..." line )to specify
// which Ps model and it's constants should be compiled. Choosing a Ps model
// automatically includes the Et and Misc constants (i.e. Gem is assumed).

{$IFDEF ArcticPs}
const
// Arctic Ps parameters
x1 = 0.192;	x2 = 0.125;
x5 = 2.196;	x6 = 50.41;
x7 = 0.161;	x8 = 14.78;
x9 = 1.146;
gammastar = 0.468;	kCO2 = 500.3;
{$ENDIF}

{$IFDEF ArcticPs OR GlobalPs}
//const
// General Et parameters
aE1 = 0.0004;    aE2 = 150;  aE3 = 1.21;   aE4 = 6.11262E5;

// Other constants
cp = 1.012E-9; //specific heat air MJ kg-1 oC-1
sigmaSB = 4.9e-9; //stefan-Boltzmann MJ m-2 day-1 K-4
S0 = 117.5; //solar constant MJ m-2 day-1
bHI1 =0.23;
bHI2 =0.48;
mw = 2.99; //kg h2o MJ-1
alphaMS = 2; //mm oC-1 day-1                                 }
{$ENDIF}

var
{ List the variable names you are going to use here. Generally, this list
  includes all the symbols you defined in the procedure counts above. The order
  in which you list them does not matter. }
{States}
Bc, dBcdt, 
Bn, dBndt, 
Vc, dVcdt, 
Vn, dVndt, 
Dc, dDcdt, 
Dn, dDndt, 
N, dNdt, 

{processes and associated parameters}
q, qWood, kq, 
GrC, 
GrN, 
aa, tau, 
LCd, 
LNd, betaD, 
LNe, betaNE, 
LCv, mC, 
LNv, mN, 
Sc, zc, zn, 
Sn, 
UCv, gC, kC, 
UNv, gN, kN, 
RCv, rC, 
RNv, rN, 
UNm, psi, alphaN, theta, kNm, 
lambdaC, epsilonC, 
lambdaN, epsilonN, 
mmC, 
mmN, 
RCm, 
RNm :double;
Vstar, a: array[1..2] of double;

{drivers}

{other}
dddd: array[1..2] of double;
Ca, 
Rne, 
o

:double; {Final double}
npar, j, jj, kk, ll, tnum:integer;

{ Functions or procedures }

begin
{ Copy the drivers from the global array, drive, into the local variables. }

{ Copy the state variables from the global array into the local variables. }
Bc := tstat[1].value;
Bn := tstat[2].value;
Vc := tstat[3].value;
Vn := tstat[4].value;
Dc := tstat[5].value;
Dn := tstat[6].value;
N := tstat[7].value;

{ And now copy the parameters into the local variables. No need to copy the
  processes from the global array into local variables. Process values will be
  calculated by this procedure.

  Copy the parameters for each process separately using the function ParCount
  to keep track of the number of parameters in the preceeding processes.
  npar now contains the number of parameters in the preceding processes.
  copy the value of the first parameter of this process into it's local
  variable }
npar:=ParCount(ModelDef.numstate + 1);
qWood := par[npar + 1].value;
kq := par[npar + 2].value;

npar:=ParCount(ModelDef.numstate + 4);
tau := par[npar + 1].value;
 
npar:=ParCount(ModelDef.numstate + 6);
betaD := par[npar + 1].value;
 
npar:=ParCount(ModelDef.numstate + 7);
betaNE := par[npar + 1].value;
 
npar:=ParCount(ModelDef.numstate + 8);
mC := par[npar + 1].value;
 
npar:=ParCount(ModelDef.numstate + 9);
mN := par[npar + 1].value;
 
npar:=ParCount(ModelDef.numstate + 10);
zc := par[npar + 1].value;
zn := par[npar + 2].value;
 
npar:=ParCount(ModelDef.numstate + 12);
gC := par[npar + 1].value;
kC := par[npar + 2].value;
 
npar:=ParCount(ModelDef.numstate + 13);
gN := par[npar + 1].value;
kN := par[npar + 2].value;
 
npar:=ParCount(ModelDef.numstate + 14);
rC := par[npar + 1].value;
 
npar:=ParCount(ModelDef.numstate + 15);
rN := par[npar + 1].value;
 
npar:=ParCount(ModelDef.numstate + 16);
psi := par[npar + 1].value;
alphaN := par[npar + 2].value;
theta := par[npar + 3].value;
kNm := par[npar + 4].value;
 
npar:=ParCount(ModelDef.numstate + 17);
epsilonC := par[npar + 1].value;
 
npar:=ParCount(ModelDef.numstate + 18);
epsilonN := par[npar + 1].value;
 
npar:=ParCount(ModelDef.numstate + 23);
for jj := 1 to 2 do a[jj] := par[npar + 1 + (jj - 1)*1].value;
 
 
{ Enter the equations to calculate the processes here, using the local variable
  names defined above. }

Q := qwood*Bc/(kq+Bc);
LCv := mC*Bc;
LNv := mN*Bn;
LNe := betaNe*N;
LCd := betaD*Dc;
LNd := betaD*Dn;
Sc := power(Bc+q*Bn, zC);
Sn := power(Bc+q*Bn, zN);
UCv := gC*Vc*Sc*Ca/(kC+Ca);
UNv := gN*VN*SN*N/(kN+N);
RCv := rc*Bc;
RNv := rn*Bn;
UNm := psi*alphaN*Dc*Dc*N/(theta*Dn*(kNm+N));
mmc:= psi*Dc;
mmn := psi*Dn + UNm;
lambdaC := epsilonC*theta*mmn/(mmc+theta*mmn);
lambdaN := epsilonN*Mmc/(mmc+theta*mmn);
RCm := Mmc*(1-lambdaC);
RNm := Mmn*(1-lambdaN);
dBndt := UNv -RNv - LNv;
dBcdt := UCv - RCv - LCv;
GrC := dBcdt/Bc;
GrN := dBndt/Bn;
Aa := ln(Bc/(q*Bn)) + tau*(GrC-GrN);
If aa>0 then Vstar := Vc else Vstar := Vn;
if CalculateDiscrete then
begin
// Add any discrete processes here
end; //discrete processes


{ Now calculate the derivatives of the state variables. If the holdConstant
  portion of the state variable is set to true then set the derivative equal to
  zero. }
if (tstat[1].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dBcdt := 0
else
 DBcdt := dBcdt;
 
if (tstat[2].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dBndt := 0
else
 dBndt := dBndt;
 
if (tstat[3].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dVcdt := 0
else
 dVcdt := -a*AA*Vstar;
 
if (tstat[4].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dVndt := 0
else
 dVndt := -dVcdt;
 
if (tstat[5].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dDcdt := 0
else
 dDcdt := LCv - LCd - Rcm;
 
if (tstat[6].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dDndt := 0
else
 dDndt := LNv + UNm - LNd - RNm;
 
if (tstat[7].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dNdt := 0
else
 dNdt := RNe + RNm +RNv-LNe-UNm-UNv;
 

{ Now that the calculations are complete, assign everything back into the arrays
  so the rest of the code can access the values calculated here. (Local variables
  are destroyed at the end of the procedure).

  Put the state variables back into the global arrays in case the state variable
  was manually changed in this procedure (e.g. discrete state variables or steady state
  calculations).   }
tstat[1].value := Bc;
tstat[2].value := Bn;
tstat[3].value := Vc;
tstat[4].value := Vn;
tstat[5].value := Dc;
tstat[6].value := Dn;
tstat[7].value := N;

{  Put all process values into process variable array. The first numstate
  processes are the derivatives of the state variables (Calculated above).}
tproc[1].value := dBcdt;
tproc[2].value := dBndt;
tproc[3].value := dVcdt;
tproc[4].value := dVndt;
tproc[5].value := dDcdt;
tproc[6].value := dDndt;
tproc[7].value := dNdt;

{ Now the remaining processes. Be sure to number the processes the same here as
  you did in the procedure counts above. }
tproc[ModelDef.numstate + 1].value := q;
tproc[ModelDef.numstate + 2].value := GrC;
tproc[ModelDef.numstate + 3].value := GrN;
tproc[ModelDef.numstate + 4].value := aa;
tproc[ModelDef.numstate + 5].value := LCd;
tproc[ModelDef.numstate + 6].value := LNd;
tproc[ModelDef.numstate + 7].value := LNe;
tproc[ModelDef.numstate + 8].value := LCv;
tproc[ModelDef.numstate + 9].value := LNv;
tproc[ModelDef.numstate + 10].value := Sc;
tproc[ModelDef.numstate + 11].value := Sn;
tproc[ModelDef.numstate + 12].value := UCv;
tproc[ModelDef.numstate + 13].value := UNv;
tproc[ModelDef.numstate + 14].value := RCv;
tproc[ModelDef.numstate + 15].value := RNv;
tproc[ModelDef.numstate + 16].value := UNm;
tproc[ModelDef.numstate + 17].value := lambdaC;
tproc[ModelDef.numstate + 18].value := lambdaN;
tproc[ModelDef.numstate + 19].value := mmC;
tproc[ModelDef.numstate + 20].value := mmN;
tproc[ModelDef.numstate + 21].value := RCm;
tproc[ModelDef.numstate + 22].value := RNm;
for jj := 1 to 2 do tproc[ModelDef.numstate + 22 + jj].value := Vstar[jj];

end;  // End of processes procedure


       { Do not make any modifications to code below this line. }
{****************************************************************************}


{This function counts the parameters in all processes less than processnum.}
function ParCount(processnum:integer) : integer;
var
 NumberofParams, counter : integer;
begin
  NumberofParams := 0;
  for counter := ModelDef.numstate + 1 to processnum - 1 do
         NumberofParams := NumberofParams + proc[counter].parameters;
  ParCount := NumberofParams;
end; // end of parcount function

{ This procedure supplies the derivatives of the state variables to the
  integrator. Since the integrator deals only with the values of the variables
  and not there names, units or the state field HoldConstant, this procedure
  copies the state values into a temporary state array and copies the value of
  HoldConstant into the temporary state array and passes this temporary state
  array to the procedure processes. }
PROCEDURE derivs(t, drt:double; var tdrive:drivearray; var tpar:paramarray;
             var statevalue:yValueArray; VAR dydt:yValueArray);
var
   i:integer;
   tempproc:processarray;
   tempstate:statearray;
begin
   tempstate := stat;  // Copy names, units and HoldConstant to tempstate
  // Copy current values of state variables into tempstate
   for i := 1 to ModelDef.numstate do tempstate[i].value := statevalue[i];
  // Calculate the process values
   processes(t, drt, tdrive, tpar, tempstate, tempproc, false);
  // Put process values into dydt array to get passed back to the integrator.
   for i:= 1 to ModelDef.numstate do dydt[i]:=tempproc[i].value;
end;  // end of derivs procedure

end.
